//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQml 2.12
import QtQuick 2.12

import wesual.Ui 1.0

FocusScope {
    id : uiSpinBox

    signal editingFinished()
    signal updateRequested(real newValue)

    /*!
    \internal
    \brief The vertical margin between label text and control border.
    */
    property int verticalMargin: 4

    /*!
    \internal
    \brief The horizontal margin between label text and control border.
    */
    property int horizontalMargin : 6

    /*!
    \internal
    \brief The opacity of the arrow buttons.
    */
    property real buttonOpacity : 1.0

    property real   value         : 0
    property real   minimumValue  : -Number.MAX_VALUE
    property real   maximumValue  : Number.MAX_VALUE
    property real   step          : 0.1
    property int    decimals      : 1
    property string unit

    property bool   bindable      : false
    property bool   indeterminate : false

    readonly property alias hovered : mouseArea.containsMouse

    function stepUp() {
        if (indeterminate)
            return;

        var newVal = Math.min(maximumValue, value + step);
        if (Math.abs(newVal - value) >= step - p_.epsilon) {
            if (!bindable) {
                value = newVal;
            } else {
                uiSpinBox.updateRequested(newVal);
            }
        }
    }

    function stepDown() {
        if (indeterminate)
            return;

        var newVal = Math.max(minimumValue, value - step);
        if (Math.abs(newVal - value) >= step - p_.epsilon) {
            if (!bindable) {
                value = newVal;
            } else {
                uiSpinBox.updateRequested(newVal);
            }
        }
    }

    QtObject {
        id : p_

        readonly property real epsilon     : .00001
        readonly property int  buttonWidth : 20
        readonly property int  spacing     :  8
        readonly property int  minTick     : 15
        readonly property int  maxTick     : 250
    }

    TextMetrics {
        id : metrics

        text : maximumValue.toFixed(decimals) + 6
        font : label.font
    }

    implicitHeight   : label.implicitHeight + 2 * verticalMargin + 1
    implicitWidth    : Math.round(label.implicitWidth + horizontalMargin
                                  + p_.buttonWidth + p_.spacing)
    activeFocusOnTab : true
    baselineOffset   : label.y + label.baselineOffset

    Keys.onUpPressed   : stepUp()
    Keys.onDownPressed : stepDown()
    onActiveFocusChanged : {
        if (activeFocus) {
            label.beginEdit();
        }
    }

    MouseArea {
        id : mouseArea

        anchors.fill : uiSpinBox
        hoverEnabled : true
        z            : -1

        Timer {
            property int nextTick : 0
            property int prevTick : p_.maxTick

            running  : mouseArea.containsPress
            interval : 10
            repeat   : true

            triggeredOnStart : true

            onRunningChanged : {
                nextTick = 0;
                prevTick = p_.maxTick;
            }

            onTriggered : {
                nextTick -= interval;

                if (nextTick >= 0)
                    return;

                for (var i = 0; i < uiSpinBox.children.length; ++i) {
                    var item = uiSpinBox.children[i];
                    var pos = mouseArea.mapToItem(
                                item, mouseArea.mouseX, mouseArea.mouseY);
                    if (item.contains(pos) && item.handleClick) {
                        item.handleClick();
                        break;
                    }
                }

                prevTick = Math.max(p_.minTick, prevTick * .8);
                nextTick = prevTick;
            }
        }

        onClicked : {
            for (var i = 0; i < uiSpinBox.children.length; ++i) {
                var item = uiSpinBox.children[i];
                var pos = mouseArea.mapToItem(
                            item, mouseArea.mouseX, mouseArea.mouseY);
                if (item.contains(pos) && item.handleClick) {
                    return;
                }
            }

            uiSpinBox.forceActiveFocus();
        }

        MouseArea {
            anchors.fill        : parent
            anchors.rightMargin : 24
            acceptedButtons     : Qt.NoButton
            cursorShape         : Qt.IBeamCursor
        }
    }

    Component {
        id : valueEditor

        TextInput {
            property bool commited : false

            font  : label.font
            anchors {
                left : label.left
                top  : label.top
            }

            function textToNumber(text) {
                const loc = Qt.locale();

                // Try parsing as-is in local format first
                try {
                    return Number.fromLocaleString(loc, text);
                } catch (err) {}

                // Try fixup - entries with wrong group separator
                // placement e.g. 100,00.00 are accepted by the validator but
                // will not parse. We try fixing up the input by removing
                // all instances of Locale.groupSeparator
                try {
                    const fixed = text.replace(loc.groupSeparator, "");
                    return Number.fromLocaleString(loc, fixed);
                } catch (err) {}

                // Try C format as a last resort
                try {
                    return Number.fromLocaleString(Qt.locale("C"), text);
                } catch (err) {}

                // At least return an valid number
                return Math.max(minimumValue, Math.min(maximumValue, 0));
            }

            function commit() {
                if (acceptableInput && !commited) {
                    var value = textToNumber(text);

                    if (!uiSpinBox.bindable) {
                        uiSpinBox.value = value;
                    } else {
                        uiSpinBox.updateRequested(value);
                    }
                    commited = true;
                }

                if (activeFocus)
                    uiSpinBox.focus = false;
                uiSpinBox.editingFinished();
                destroy();
            }

            function discard() {
                text = "";
                if (activeFocus)
                    uiSpinBox.focus = false;
            }

            color  : UiColors.getColor(UiColors.Black)

            selectByMouse     : true
            selectionColor    : UiColors.getColor(UiColors.SelectionGreen)
            selectedTextColor : UiColors.getColor(UiColors.White)

            Component.onCompleted : {
                forceActiveFocus();
                selectAll();
            }
            Keys.onEnterPressed  : commit()
            Keys.onReturnPressed : commit()
            Keys.onEscapePressed : discard()
            onActiveFocusChanged : {
                if (!activeFocus) {
                    commit();
                }
            }

            validator : DoubleValidator {
                bottom   : uiSpinBox.minimumValue
                top      : uiSpinBox.maximumValue
                decimals : uiSpinBox.decimals
                locale   : Qt.locale().name
                notation : DoubleValidator.StandardNotation
            }
        }
    }

    UiText {
        id : label

        function beginEdit() {
            if (editor)
                return;
            editor = valueEditor.createObject(uiSpinBox, {
                "text" : indeterminate ? "" : label.text
            });
        }

        property TextInput editor : null

        text    : {
            var label = uiSpinBox.value.toLocaleString(
                        Qt.locale(), "f", uiSpinBox.decimals);

            return label;
        }
        visible : !editor && !uiSpinBox.indeterminate

        onTextChanged : {
            if (editor) {
                editor.text = text;
            }
        }

        anchors {
            left           : uiSpinBox.left
            leftMargin     : uiSpinBox.horizontalMargin
            verticalCenter : uiSpinBox.verticalCenter
        }

        UiText {
            id : unitText

            anchors {
                left     : label.right
                baseline : label.baseline
            }
            text    : uiSpinBox.unit
            visible : uiSpinBox.unit.length > 0
        }
    }

    Rectangle {
        id : up

        readonly property bool hovered :
            mouseArea.containsMouse && contains(mapFromItem(
                mouseArea, mouseArea.mouseX, mouseArea.mouseY))
        readonly property bool pressed : hovered && mouseArea.pressed

        function handleClick() {
            uiSpinBox.stepUp();
        }

        opacity : uiSpinBox.buttonOpacity

        states : [
            State {
                name : "disabled"
                when : !uiSpinBox.enabled || uiSpinBox.indeterminate ||
                       (uiSpinBox.maximumValue - uiSpinBox.value) < p_.epsilon

                PropertyChanges {
                    target : arrowUp
                    color  : UiColors.getColor(UiColors.DisabledLightGrey13)
                }
            },
            State {
                name : "pressed"
                when : up.pressed

                PropertyChanges {
                    target : up
                    color  : UiColors.getColor(UiColors.SelectionGreen)
                }
                PropertyChanges {
                    target : arrowUp
                    color  : UiColors.getColor(UiColors.White)
                }
            },
            State {
                name : "hovered"
                when : up.hovered

                PropertyChanges {
                    target : up
                    color  : UiColors.getColor(UiColors.HoverGreen)
                }
            }
        ]

        width  : p_.buttonWidth
        height : 11
        color  : UiColors.getColor(UiColors.White)
        anchors {
            right   : uiSpinBox.right
            top     : uiSpinBox.top
            margins : 1
        }
        UiColorizedImage {
            id : arrowUp

            source : "qrc:/ui/images/spinbox-arrow-up.png"
            color  : "transparent"
            x : 6
            anchors {
                bottom       : up.bottom
                bottomMargin : 3
            }
        }
    }
    Rectangle {
        id : down

        readonly property bool hovered :
            mouseArea.containsMouse && contains(mapFromItem(
                mouseArea, mouseArea.mouseX, mouseArea.mouseY))
        readonly property bool pressed : hovered && mouseArea.pressed

        function handleClick() {
            uiSpinBox.stepDown();
        }

        opacity : uiSpinBox.buttonOpacity

        states : [
            State {
                name : "disabled"
                when : !uiSpinBox.enabled || uiSpinBox.indeterminate ||
                       (uiSpinBox.value - uiSpinBox.minimumValue) < p_.epsilon

                PropertyChanges {
                    target : arrowDown
                    color  : UiColors.getColor(UiColors.DisabledLightGrey13)
                }
            },
            State {
                name : "pressed"
                when : down.pressed

                PropertyChanges {
                    target : down
                    color  : UiColors.getColor(UiColors.SelectionGreen)
                }
                PropertyChanges {
                    target : arrowDown
                    color  : UiColors.getColor(UiColors.White)
                }
            },
            State {
                name : "hovered"
                when : down.hovered

                PropertyChanges {
                    target : down
                    color  : UiColors.getColor(UiColors.HoverGreen)
                }
            }
        ]

        width  : p_.buttonWidth
        height : 11
        anchors {
            right   : uiSpinBox.right
            bottom  : uiSpinBox.bottom
            margins : 1
        }
        UiColorizedImage {
            id : arrowDown

            source : "qrc:/ui/images/spinbox-arrow-down.png"
            color  : "transparent"
            x : 6
            anchors {
                top       : down.top
                topMargin : 3
            }
        }
    }
}

